var FishManager = pc.createScript('fishManager');

// Entité ou prefab pour le poisson
FishManager.attributes.add('fishPrefab', {
    type: 'entity',
    title: 'Fish Prefab'
});

// Taux d'émission (poissons par seconde)
FishManager.attributes.add('fishPerSecond', {
    type: 'number',
    default: 2,
    title: 'Poissons / Seconde'
});

// Durée de vie des poissons (en secondes)
// Si = 0, on peut considérer que les poissons ne meurent jamais
FishManager.attributes.add('fishLifetime', {
    type: 'number',
    default: 10,
    title: 'Durée de Vie (s)',
    description: 'Temps après lequel un poisson sera détruit'
});

// Zone de spawn (entité sol/plane ou autre) pour calculer la bounding box
FishManager.attributes.add('groundEntity', {
    type: 'entity',
    title: 'Ground Entity'
});

// Zone d’exclusion (optionnelle)
FishManager.attributes.add('exclusionZone', {
    type: 'entity',
    title: 'Exclusion Zone',
    description: 'Entité d’exclusion'
});

FishManager.prototype.initialize = function() {
    // Accumulateur de temps pour savoir quand émettre un nouveau poisson
    this.spawnAccumulator = 0;

    // Liste des poissons en cours dans la scène
    // On stocke { entity: <entitéPoisson>, birthTime: <time écoulé> }
    this.activeFishes = [];

    // On récupère l'AABB (bounding box) de la zone de spawn
    this.groundAabb = this.getCombinedAabbFromEntity(this.groundEntity);
    if (!this.groundAabb) {
        console.warn('[FishManager] Impossible de récupérer le bounding box du sol.');
    }

    // Temps écoulé global, pour calculer l'âge de chaque poisson
    this.timeSinceStart = 0;
};

FishManager.prototype.update = function(dt) {
    // Incrémente le temps écoulé global
    this.timeSinceStart += dt;

    // Si on n'a pas de zone de spawn ou pas de prefab, on arrête
    if (!this.groundAabb || !this.fishPrefab) return;

    // ─────────────────────────────────────────────────────────────────────
    // 1) Gestion de l'émission (spawn) des poissons
    // ─────────────────────────────────────────────────────────────────────

    // On ajoute le dt à notre accumulateur
    this.spawnAccumulator += dt;

    // Combien de poissons doit-on émettre par seconde ?
    // => Intervalle entre 2 spawns = (1 / fishPerSecond)
    var spawnInterval = 0;
    if (this.fishPerSecond > 0) {
        spawnInterval = 1 / this.fishPerSecond;
    } else {
        spawnInterval = Infinity; // s'il est à 0, on n'émet rien
    }

    // Tant qu'on dépasse l'intervalle, on émet un poisson
    while (this.spawnAccumulator >= spawnInterval) {
        this.spawnAccumulator -= spawnInterval;
        this.spawnOneFish();
    }

    // ─────────────────────────────────────────────────────────────────────
    // 2) Gestion de la destruction (durée de vie)
    // ─────────────────────────────────────────────────────────────────────
    if (this.fishLifetime > 0) {
        // On parcourt la liste en sens inverse pour pouvoir retirer sans soucis
        for (var i = this.activeFishes.length - 1; i >= 0; i--) {
            var fishInfo = this.activeFishes[i];
            var age = this.timeSinceStart - fishInfo.birthTime;
            // Si l'âge dépasse la durée de vie
            if (age >= this.fishLifetime) {
                // On détruit le poisson de la scène
                fishInfo.entity.destroy();
                // Et on le retire du tableau
                this.activeFishes.splice(i, 1);
            }
        }
    }
};

/**
 * Instancie un poisson (si possible), positionne hors zone d’exclusion,
 * puis l'ajoute à la scène.
 */
FishManager.prototype.spawnOneFish = function() {
    // Clone du prefab
    var newFish = this.fishPrefab.clone();
    newFish.enabled = true;

    // On trouve une position valide (en dehors de l’exclusion)
    var randomPos = null;
    var maxTry = 30;
    for (var t = 0; t < maxTry; t++) {
        randomPos = this.getRandomPositionInAabb(this.groundAabb);
        if (!this.isInsideExclusionZone(randomPos)) {
            break;
        }
    }

    // Placer le poisson (ex. à la hauteur max du sol ou min, selon le besoin)
    var y = this.groundAabb.getMax().y; // On prend le "top" du bounding box
    newFish.setPosition(randomPos.x, y, randomPos.z);

    // Ajout à la scène
    this.app.root.addChild(newFish);

    // On enregistre l'entité et son "birthTime"
    this.activeFishes.push({
        entity: newFish,
        birthTime: this.timeSinceStart
    });
};

// ─────────────────────────────────────────────────────────────────────────────
// ─────────── OUTILS (Bounding Box / Random / Exclusion) ─────────────────────
// ─────────────────────────────────────────────────────────────────────────────

/**
 * Récupère le bounding box mondial (AABB) “global” d’une entité
 * Render ou Model, en combinant toutes ses meshInstances si besoin.
 */
FishManager.prototype.getCombinedAabbFromEntity = function(entity) {
    if (!entity) return null;

    var meshInstances = null;

    // RenderComponent (nouveau système)
    if (entity.render && entity.render.meshInstances) {
        meshInstances = entity.render.meshInstances;
    }
    // ModelComponent (ancien système)
    else if (entity.model && entity.model.meshInstances) {
        meshInstances = entity.model.meshInstances;
    }

    if (!meshInstances || meshInstances.length === 0) {
        return null;
    }

    var combinedAabb = null;
    for (var i = 0; i < meshInstances.length; i++) {
        var mi = meshInstances[i];
        mi.syncAabb(); // mise à jour
        if (!combinedAabb) {
            combinedAabb = mi.aabb.clone();
        } else {
            combinedAabb.add(mi.aabb);
        }
    }
    return combinedAabb;
};

/**
 * Renvoie une position (x, 0, z) aléatoire
 * à l’intérieur de l’AABB donné (en XZ).
 * On fixera Y plus tard.
 */
FishManager.prototype.getRandomPositionInAabb = function(aabb) {
    var min = aabb.getMin();
    var max = aabb.getMax();

    var x = pc.math.random(min.x, max.x);
    var z = pc.math.random(min.z, max.z);

    return new pc.Vec3(x, 0, z);
};

/**
 * Vérifie si posXZ est DANS la zone d’exclusion (si définie).
 */
FishManager.prototype.isInsideExclusionZone = function(posXZ) {
    if (!this.exclusionZone) return false;

    var exclusionAabb = this.getCombinedAabbFromEntity(this.exclusionZone);
    if (!exclusionAabb) return false;

    var minX = exclusionAabb.getMin().x;
    var maxX = exclusionAabb.getMax().x;
    var minZ = exclusionAabb.getMin().z;
    var maxZ = exclusionAabb.getMax().z;

    return (
        posXZ.x >= minX && posXZ.x <= maxX &&
        posXZ.z >= minZ && posXZ.z <= maxZ
    );
};
