var shaderMultivers = pc.createScript('shaderMultivers');

// Attribut pour sélectionner, dans l'éditeur, le matériau cible à remplacer
shaderMultivers.attributes.add('targetMaterial', {
    type: 'asset',
    assetType: 'material',
    title: 'Target Material'
});

// Si tu souhaites fournir une texture "iChannel1" comme dans Shadertoy,
// tu peux éventuellement décommenter la ligne ci-dessous et associer un asset "texture".
// shaderMultivers.attributes.add('iChannel1', {
//     type: 'asset',
//     assetType: 'texture',
//     title: 'Texture iChannel1'
// });

shaderMultivers.prototype.initialize = function() {
    var device = this.app.graphicsDevice;

    // Vérifie qu'un matériau cible a été assigné
    if (!this.targetMaterial || !this.targetMaterial.resource) {
        console.warn("shaderMultivers: Pas de 'Target Material' assigné !");
        return;
    }

    // --- Vertex Shader ---
    var vs = `
attribute vec3 aPosition;
attribute vec2 aUv0;
uniform mat4 matrix_model;
uniform mat4 matrix_viewProjection;
varying vec2 vUv0;

void main(void) {
    vUv0 = aUv0;
    gl_Position = matrix_viewProjection * matrix_model * vec4(aPosition, 1.0);
}
    `;

    // --- Fragment Shader ---
    // Adaptation du code Shadertoy. On y inclut iTime -> "time", iResolution, et iChannel1 (optionnel).
    var fs = `
precision ${device.precision} float;
varying vec2 vUv0;

// Uniforms (PlayCanvas)
uniform float iTime;        // "time" côté Shadertoy
uniform vec2 iResolution;   // Résolution
uniform sampler2D iChannel1; // Optionnel, si tu veux lier une texture

//-----------------CONSTANTS MACROS-----------------
#define PI 3.14159265359
#define E 2.7182818284
#define GR 1.61803398875

//-----------------UTILITY MACROS-----------------

// Note : __LINE__ renverra un numéro de ligne différent qu'en Shadertoy, mais on conserve le principe
#define time (sin(((sin(float(__LINE__)*100.0)*GR/PI+GR/PI/E)*iTime+100.0)/100.0)*100.0)
#define saw(x) (acos(cos(x))/PI)
#define sphereN(uv) (clamp(1.0-length(uv*2.0-1.0), 0.0, 1.0))
#define clip(x) (smoothstep(0.5-GR/PI/E, .5+GR/PI/E, x))
#define zero(x) (smoothstep(-1.0/PI/E/GR, 1.0/PI/E/GR, sin(x*PI/2.0))*2.0-1.0)
#define TIMES_DETAILED (1.0)
#define angle(uv) (atan((uv).y, (uv).x))
#define angle_percent(uv) ((angle(uv)/PI+1.0)/2.0)
#define absMin(x,y) (abs(x) < abs(y) ? x: y)
#define quadrant(uv) (absMin((zero(uv).x), (zero(uv).y))+floor(uv.x/2.0)+floor(uv.y/2.0))

#define flux(x) (vec3(cos(x),cos(4.0*PI/3.0+x),cos(2.0*PI/3.0+x))*.5+.5)
#define rotatePoint(p,n,theta) (p*cos(theta)+cross(n,p)*sin(theta)+n*dot(p,n)*(1.0-cos(theta)))
#define GUASS(x) (smoothstep(0.0, 1.0/GR/PI/E, saw(x*PI/2.0)*(1.0-saw(x*PI/2.0))))

#define GRID_COUNT (50.0)
#define hash(p) (fract(sin(vec2( dot(p,vec2(127.5,313.7)),dot(p,vec2(239.5,185.3))))*43458.3453))

#define MAX_DIM (max(iResolution.x, iResolution.y))

float seedling = 0.0;

vec2 spiral(vec2 uv)
{
    float turns = 5.0;
    float r = pow(log(length(uv)+1.0), 1.175);
    float theta = atan(uv.y, uv.x)*turns - r*PI;
    return vec2(saw(r*PI + iTime), saw(theta + iTime*1.1));
}

vec2 cmul(vec2 v1, vec2 v2) {
    return vec2(
        v1.x * v2.x - v1.y * v2.y,
        v1.y * v2.x + v1.x * v2.y
    );
}

vec2 cdiv(vec2 v1, vec2 v2) {
    return vec2(
        v1.x * v2.x + v1.y * v2.y,
        v1.y * v2.x - v1.x * v2.y
    ) / dot(v2, v2);
}

vec2 mobius(vec2 uv, vec2 multa, vec2 offa, vec2 multb, vec2 offb)
{
    return saw(cdiv(cmul(uv, multa) + offa, cmul(uv, multb) + offb)*PI)*2.0-1.0;
}

vec2 square_map(vec2 uv)
{
    return (
        rotatePoint(
            vec3(uv + vec2(cos(seedling*PI), cos(seedling*GR)), 0.0),
            vec3(0.0, 0.0, 1.0),
            time/PI
        ).xy * (1.0 + sin(time + seedling)/PI/E/GR)
        + vec2(cos(time + seedling) + sin(time + seedling))
    );
}

vec2 iterate_square(vec2 uv, vec2 dxdy, out float magnification)
{
    vec2 a = uv + vec2(0.0,       0.0);
    vec2 b = uv + vec2(dxdy.x,    0.0);
    vec2 c = uv + vec2(dxdy.x,    dxdy.y);
    vec2 d = uv + vec2(0.0,       dxdy.y);

    vec2 ma = square_map(a);
    vec2 mb = square_map(b);
    vec2 mc = square_map(c);
    vec2 md = square_map(d);

    float da = length(mb - ma);
    float db = length(mc - mb);
    float dc = length(md - mc);
    float dd = length(ma - md);

    float stretch = max(
        max(
            max(da/dxdy.x, db/dxdy.y),
            dc/dxdy.x
        ),
        dd/dxdy.y
    );

    magnification = stretch;

    return square_map(uv);
}

vec2 mobius_map(vec2 uv, vec2 multa, vec2 offa, vec2 multb, vec2 offb)
{
    return mobius(uv, multa, offa, multb, offb);
}

vec2 iterate_mobius(vec2 uv, vec2 dxdy, out float magnification,
                    vec2 multa, vec2 offa, vec2 multb, vec2 offb)
{
    vec2 a = uv + vec2(0.0,       0.0);
    vec2 b = uv + vec2(dxdy.x,    0.0);
    vec2 c = uv + vec2(dxdy.x,    dxdy.y);
    vec2 d = uv + vec2(0.0,       dxdy.y);

    vec2 ma = mobius_map(a, multa, offa, multb, offb);
    vec2 mb = mobius_map(b, multa, offa, multb, offb);
    vec2 mc = mobius_map(c, multa, offa, multb, offb);
    vec2 md = mobius_map(d, multa, offa, multb, offb);

    float da = length(mb - ma);
    float db = length(mc - mb);
    float dc = length(md - mc);
    float dd = length(ma - md);

    float stretch = max(
        max(
            max(da/dxdy.x, db/dxdy.y),
            dc/dxdy.x
        ),
        dd/dxdy.y
    );

    magnification = stretch;

    return mobius_map(uv, multa, offa, multb, offb);
}

vec3 phase(float m)
{
    return vec3(
        saw(m),
        saw(4.0*PI/3.0 + m),
        saw(2.0*PI/3.0 + m)
    );
}

vec3 hash3(vec2 p)
{
    vec3 q = vec3(
        dot(p, vec2(123.4, 234.5)),
        dot(p, vec2(456.7, 321.0)),
        dot(p, vec2(432.1, 543.2))
    );
    return fract(sin(q)*12345.678);
}

vec4 galaxy(vec2 uv)
{
    uv *= 5.0;

    float r1 = log(length(uv) + 1.0)*2.0;
    float r2 = pow(log(length(uv) + 1.0)*3.0, 0.5);

    float rotation = time;
    float theta1 = atan(uv.y, uv.x) - r1*PI + rotation*0.5 + seedling;
    float theta2 = atan(uv.y, uv.x) - r2*PI + rotation*0.5 + seedling;

    vec4 color = vec4(flux(time + seedling), 1.0);

    // Quelques combinaisons cos/sin
    vec4 final = acos(1.0 - (cos(theta1)*cos(theta1) + sqrt(cos(theta1 + PI)*cos(theta1 + PI))) / 2.0)
                 * (1.0 - log(r1 + 1.0)) * color
               + cos(1.0 - (cos(theta2)*cos(theta2) + cos(theta2 + PI/2.0)*cos(theta2 + PI/2.0)) / 2.0)
                 * (1.25 - log(r2 + 1.0)) * color;

    final.rgba += color;
    final /= r1;
    final *= 2.0;

    return final;
}

void mainImage(out vec4 fragColor, in vec2 fragCoord)
{
    // Lecture d'une éventuelle texture (optionnelle)
    // vec4 sample1 = texture2D(iChannel1, fragCoord / iResolution.xy);

    // Coordonnées uv entre [-1, 1] (à peu près)
    vec2 uv = fragCoord.xy / iResolution.xy;

    float scale = exp(sin(time))*E + GR;
    uv = uv*scale - scale/2.0;
    uv.x *= iResolution.x / iResolution.y;
    uv = rotatePoint(vec3(uv, 0.0), vec3(0.0, 0.0, 1.0), time/PI).xy;

    vec2 uv0 = uv;
    uv += cos(vec2(time, time/GR));
    float r = length(uv);

    float map = time;
    float noise = 1.0;
    float spounge = time*4.0*PI;
    const int max_iterations = 4;

    vec2 multa, multb, offa, offb;
    float antispeckle = 1.0; 
    float magnification = 1.0;
    
    vec3 color = vec3(1.0);
    vec3 accum = vec3(0.0);
    float sum = 0.0;
    float anticolor = 1.0;
    seedling = 0.0;

    float border = 0.0;

    vec4 hit = vec4(0.0);

    // Boucle d'itération principale
    for(int i = 0; i < max_iterations; i++)
    {
        float iteration = float(i)/float(max_iterations);

        // Prépare des "paramètres" pour mobius/square_map
        multa = cos(vec2(time*1.1, time*1.2) + iteration*PI);
        offa  = cos(vec2(time*1.3, time*1.4) + iteration*PI)*PI;
        multb = cos(vec2(time*1.5, time*1.6) + iteration*PI);
        offb  = cos(vec2(time*1.7, time*1.8) + iteration*PI);

        // 1) Application du square_map + calcul du "stretch"
        uv = iterate_square(uv, 0.5/iResolution.xy, magnification);
        antispeckle *= smoothstep(0.0, 1.0/TIMES_DETAILED, sqrt(1.0/(1.0+magnification)));

        float q = quadrant(uv);
        seedling += q + float(i);

        map += (q + seedling)*antispeckle;
        float shift = time;

        // Contrôle d'un border
        border = max(border, (smoothstep(1.0-1.0/GR/E/PI, 1.0, cos(uv.y*PI))));
        border = max(border, (smoothstep(1.0-1.0/GR/E/PI, 1.0, cos(uv.x*PI))));

        float stripes = map*1.0*PI;
        float black = smoothstep(0.0, 0.75, saw(stripes)) * clamp(1.0 - abs(border), 0.0, 1.0);
        float white = smoothstep(0.75, 1.0, saw(stripes)) * black;

        vec3 final = flux(map*2.0*PI + shift + float(i))*black + white;

        color *= final;
        accum += final;
        sum += 1.0;
        anticolor *= white;

        // 2) Deuxième transformation (Mobius)
        if(i != 0) {
            hit += galaxy(saw(uv*PI/2.0)*2.0 - 1.0)
                   * clamp(1.0 - length(hit.rgb), 0.0, 1.0)
                   * (1.0 - border);

            uv = iterate_mobius(uv, 0.5/iResolution.xy, magnification, multa, offa, multb, offb);
            antispeckle *= smoothstep(0.0, 1.0/TIMES_DETAILED, sqrt(1.0/(1.0+magnification)));

            border = max(border, (smoothstep(1.0-1.0/GR/E/PI, 1.0, cos(uv.y*PI))));
            border = max(border, (smoothstep(1.0-1.0/GR/E/PI, 1.0, cos(uv.x*PI))));
        }
    }

    // Petites étoiles sur un "grid"
    float starScale = 32.0;
    vec2 gridPosition = floor(uv0 * starScale) / starScale;
    vec2 randomOffset = hash(gridPosition)*2.0 - 1.0;
    vec2 localGridPositionCenter = fract(uv0 * starScale) - 0.5;
    float stars = mix(0.0, 1.0, step(length(localGridPositionCenter + randomOffset*0.5), 0.1));

    float map2 = (stars + length(randomOffset)) * PI * 2.0;
    float twinkle = saw(time + map2);

    // Ajout au "hit"
    hit += clamp(
        vec4(flux(map2 + time)*PI + twinkle, 1.0)*stars*twinkle*PI,
        0.0,
        1.0
    ) * clamp(1.0 - border, 0.0, 1.0);

    // Final
    color = pow(color, vec3(1.0 / float(max_iterations)));
    antispeckle = pow(antispeckle, 1.0 / float(max_iterations));

    vec4 outColor = vec4((color + accum/sum)*(1.0 - border), 1.0);

    // Ici, on choisit de remplacer la sortie par "hit" pour privilégier l'effet Galaxy,
    // ou de faire un mix. Dans le code source, la dernière ligne est "fragColor = hit;".
    // On suit exactement le code Shadertoy d'origine.
    fragColor = hit;
}

void main(void)
{
    // Coordonées pixel
    vec2 fragCoord = vUv0 * iResolution;
    vec4 color = vec4(0.0);
    mainImage(color, fragCoord);
    gl_FragColor = color;
}
    `;

    // Définit le shader
    var shaderDefinition = {
        attributes: {
            aPosition: pc.SEMANTIC_POSITION,
            aUv0: pc.SEMANTIC_TEXCOORD0
        },
        vshader: vs,
        fshader: fs
    };

    // Crée le shader et le matériau
    this.shader = new pc.Shader(device, shaderDefinition);
    this.shaderMaterial = new pc.Material();
    this.shaderMaterial.shader = this.shader;
    this.shaderMaterial.cull = pc.CULLFACE_NONE; // Désactive éventuellement le face culling
    this.shaderMaterial.update();

    // Initialise les uniforms
    this.time = 0;
    this.shaderMaterial.setParameter("iTime", 0.0);
    this.shaderMaterial.setParameter("iResolution", [device.width, device.height]);

    // Si on utilise vraiment une texture iChannel1, on la lie ici :
    // if (this.iChannel1 && this.iChannel1.resource) {
    //     this.shaderMaterial.setParameter("iChannel1", this.iChannel1.resource);
    // }

    this.shaderMaterial.update();

    // Remplace le matériau
    var self = this;
    this.app.root.find(function(node) {
        if (node.model && node.model.meshInstances) {
            node.model.meshInstances.forEach(function(mi) {
                if (mi.material === self.targetMaterial.resource) {
                    mi.material = self.shaderMaterial;
                }
            });
        } else if (node.render && node.render.meshInstances) {
            node.render.meshInstances.forEach(function(mi) {
                if (mi.material === self.targetMaterial.resource) {
                    mi.material = self.shaderMaterial;
                }
            });
        }
    });
};

shaderMultivers.prototype.update = function(dt) {
    // Mise à jour du temps
    this.time += dt;
    this.shaderMaterial.setParameter("iTime", this.time);

    // Mise à jour de la résolution en cas de resize
    var device = this.app.graphicsDevice;
    this.shaderMaterial.setParameter("iResolution", [device.width, device.height]);

    this.shaderMaterial.update();
};
