var shaderSky1 = pc.createScript('shaderSky1');

// Attribut pour sélectionner, dans l'éditeur, le matériau cible à remplacer
shaderSky1.attributes.add('targetMaterial', {
    type: 'asset',
    assetType: 'material',
    title: 'Target Material'
});

shaderSky1.prototype.initialize = function() {
    var device = this.app.graphicsDevice;

    // Vérifie qu'un matériau cible a été assigné
    if (!this.targetMaterial || !this.targetMaterial.resource) {
        console.warn("shaderSky1: Pas de 'Target Material' assigné !");
        return;
    }

    // --- Vertex Shader ---
    var vs = `
attribute vec3 aPosition;
attribute vec2 aUv0;
uniform mat4 matrix_model;
uniform mat4 matrix_viewProjection;
varying vec2 vUv0;

void main(void) {
    vUv0 = aUv0;
    gl_Position = matrix_viewProjection * matrix_model * vec4(aPosition, 1.0);
}
    `;

    // --- Fragment Shader ---
    var fs = `
precision ${device.precision} float;
varying vec2 vUv0;

// Uniforms
uniform float time;         // iTime dans Shadertoy
uniform vec2 iResolution;   // Résolution de l'écran

// Adaptation du code Shadertoy
#define NS 100.0
#define CI 0.3

float N21(vec2 p) {
    return fract(sin(p.x * 100.0 + p.y * 7446.0) * 8345.0);
}

float SS(vec2 uv) {
    vec2 lv = fract(uv);
    // "smooth interpolation" type Shadertoy
    lv = lv * lv * (3.0 - 2.0 * lv);
    vec2 id = floor(uv);

    float bl = N21(id);
    float br = N21(id + vec2(1.0, 0.0));
    float b = mix(bl, br, lv.x);

    float tl = N21(id + vec2(0.0, 1.0));
    float tr = N21(id + vec2(1.0, 1.0));
    float t = mix(tl, tr, lv.x);

    return mix(b, t, lv.y);
}

float L(vec2 uv, vec2 ofs, float b, float l) {
    return smoothstep(0.0, 1000.0, b * max(0.1, l) / 
                      pow(max(1e-13, length(uv - ofs)), 1.0 / max(0.1, l)));
}

float rand(vec2 co, float s){
    float PHI = 1.6180339887498948482;
    return fract(tan(distance(co * PHI, co) * s) * co.x);
}

vec2 H12(float s) {
    float x = rand(vec2(243.234, 63.834), s) - 0.5;
    float y = rand(vec2(53.1434, 13.1234), s) - 0.5;
    return vec2(x, y);
}

// Équivalent de mainImage(...), version Shadertoy
void mainImage(out vec4 fragColor, in vec2 fragCoord) {
    vec2 uv = fragCoord / iResolution.xy;
    uv -= 0.5;
    uv.x *= iResolution.x / iResolution.y;

    vec4 col = vec4(0.0);

    // Couleurs de base
    vec4 b  = vec4(0.01176470588, 0.05098039215, 0.14117647058, 1.0);
    vec4 p  = vec4(0.13333333333, 0.07843137254, 0.13725490196, 1.0);
    vec4 lb = vec4(0.10196078431, 0.21568627451, 0.33333333333, 1.0);

    // Dégradé de base
    vec4 blb = mix(b, lb, -uv.x * 0.2 - (uv.y * 0.5));
    col += mix(blb, p, uv.x - (uv.y * 1.5));

    // Petits "points" (étoiles) avec un loop
    for(float i = 0.0; i < NS; i++) {
        vec2 ofs = H12(i + 1.0);
        ofs *= vec2(1.8, 1.1);

        // Variation de taille toutes les 20 "étoiles"
        float r = (mod(i, 20.0) == 0.0) ? 0.5 + abs(sin(i / 50.0)) : 0.25;

        // Effet de scintillement : sin(fract(time) * 0.5 * i)
        float flicker = (sin(fract(time) * 0.5 * i) + 1.0) * 0.02;

        col += vec4(L(uv, ofs, r + flicker, 1.0));
    }

    // Légère animation horizontale et verticale du fond
    uv.x += time * 0.03;
    uv.y += sin(time * 0.03);

    // Bruit fractal simple
    float c = 0.0;
    for(float i = 1.0; i < 8.0; i += 1.0) {
        c += SS(uv * pow(2.0, i)) * pow(0.5, i);
    }

    // Teinte finale (CI = 0.3)
    col += c * CI;

    fragColor = col;
}

void main(void) {
    // Conversion des UV en coordonnées pixel
    vec2 fragCoord = vUv0 * iResolution;
    vec4 color = vec4(0.0);
    mainImage(color, fragCoord);
    gl_FragColor = color;
}
    `;

    // Définition du shader
    var shaderDefinition = {
        attributes: {
            aPosition: pc.SEMANTIC_POSITION,
            aUv0: pc.SEMANTIC_TEXCOORD0
        },
        vshader: vs,
        fshader: fs
    };

    // Création du shader et du matériau
    this.shader = new pc.Shader(device, shaderDefinition);
    this.shaderMaterial = new pc.Material();
    this.shaderMaterial.shader = this.shader;
    this.shaderMaterial.cull = pc.CULLFACE_NONE; // désactive éventuellement le face culling
    this.shaderMaterial.update();

    // Initialisation des uniforms
    this.time = 0;
    this.shaderMaterial.setParameter("time", 0.0);
    this.shaderMaterial.setParameter("iResolution", [device.width, device.height]);
    this.shaderMaterial.update();

    // Remplacement du matériau dans la scène
    var self = this;
    this.app.root.find(function(node) {
        if (node.model && node.model.meshInstances) {
            node.model.meshInstances.forEach(function(mi) {
                if (mi.material === self.targetMaterial.resource) {
                    mi.material = self.shaderMaterial;
                }
            });
        } else if (node.render && node.render.meshInstances) {
            node.render.meshInstances.forEach(function(mi) {
                if (mi.material === self.targetMaterial.resource) {
                    mi.material = self.shaderMaterial;
                }
            });
        }
    });
};

shaderSky1.prototype.update = function(dt) {
    this.time += dt;
    this.shaderMaterial.setParameter("time", this.time);

    var device = this.app.graphicsDevice;
    this.shaderMaterial.setParameter("iResolution", [device.width, device.height]);

    this.shaderMaterial.update();
};
