var shaderFallingStripes = pc.createScript('shaderFallingStripes');

// Attribut pour sélectionner, dans l'éditeur, le matériau cible à remplacer
shaderFallingStripes.attributes.add('targetMaterial', {
    type: 'asset',
    assetType: 'material',
    title: 'Target Material'
});

shaderFallingStripes.prototype.initialize = function() {
    var device = this.app.graphicsDevice;

    // Vérifie qu'un matériau cible a été assigné
    if (!this.targetMaterial || !this.targetMaterial.resource) {
        console.warn("shaderFallingStripes: Pas de 'Target Material' assigné !");
        return;
    }

    // --- Vertex Shader ---
    var vs = `
attribute vec3 aPosition;
attribute vec2 aUv0;
uniform mat4 matrix_model;
uniform mat4 matrix_viewProjection;
varying vec2 vUv0;

void main(void) {
    vUv0 = aUv0;
    gl_Position = matrix_viewProjection * matrix_model * vec4(aPosition, 1.0);
}
    `;

    // --- Fragment Shader ---
    // Adaptation du code Shadertoy
    var fs = `
precision ${device.precision} float;
varying vec2 vUv0;

// Uniforms
uniform float time;         // iTime dans Shadertoy
uniform vec2 iResolution;   // Résolution de l'écran

// Constantes
#define FALLING_SPEED  0.25
#define STRIPES_FACTOR 5.0

// Fonction auxiliaire : "sphere" (pour l'effet de glow)
float sphere(vec2 coord, vec2 pos, float r) {
    vec2 d = pos - coord; 
    return smoothstep(60.0, 0.0, dot(d, d) - r * r);
}

// Equivalent "mainImage" de Shadertoy
void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    // Normalisation des coordonnées pixel
    vec2 uv = fragCoord / iResolution.xy;
    
    // Pixellisation en X (STRIPES_FACTOR)
    vec2 clamped_uv = (floor(fragCoord / STRIPES_FACTOR) * STRIPES_FACTOR) / iResolution.xy;
    
    // Pseudo-random basé sur clamped_uv.x (pour la hauteur des bandes)
    float value = fract(sin(clamped_uv.x) * 43758.5453123);
    
    // Création des bandes verticales
    vec3 col = vec3(1.0 - mod(uv.y * 0.5 + (time * (FALLING_SPEED + value / 5.0)) + value, 0.5));
    
    // Ajout d'une modulation colorée
    col *= clamp(cos(time * 2.0 + uv.xyx + vec3(0.0, 2.0, 4.0)), 0.0, 1.0);
    
    // Ajout des "glowing ends" (sphères)
    col += vec3(
        sphere(
            fragCoord,
            vec2(
                clamped_uv.x,
                (1.0 - 2.0 * mod((time * (FALLING_SPEED + value / 5.0)) + value, 0.5))
            ) * iResolution.xy,
            0.9
        )
    ) / 2.0;
    
    // Léger fondu de l'écran en haut/bas
    col *= vec3(exp(-pow(abs(uv.y - 0.5), 6.0) / pow(2.0 * 0.05, 2.0)));
    
    // Sortie
    fragColor = vec4(col, 1.0);
}

void main(void) {
    // On appelle mainImage en lui passant la coordonnée pixel
    vec2 fragCoord = vUv0 * iResolution;
    vec4 color = vec4(0.0);
    mainImage(color, fragCoord);
    gl_FragColor = color;
}
    `;

    // Création du shader
    var shaderDefinition = {
        attributes: {
            aPosition: pc.SEMANTIC_POSITION,
            aUv0: pc.SEMANTIC_TEXCOORD0
        },
        vshader: vs,
        fshader: fs
    };
    this.shader = new pc.Shader(device, shaderDefinition);

    // Création du matériau custom
    this.shaderMaterial = new pc.Material();
    this.shaderMaterial.shader = this.shader;
    this.shaderMaterial.cull = pc.CULLFACE_NONE; // Désactive éventuellement le face culling
    this.shaderMaterial.update();

    // Initialisation des uniforms
    this.time = 0;
    this.shaderMaterial.setParameter("time", 0.0);
    this.shaderMaterial.setParameter("iResolution", [device.width, device.height]);
    this.shaderMaterial.update();

    // Remplace le matériau cible par notre matériau custom
    var self = this;
    this.app.root.find(function(node) {
        if (node.model && node.model.meshInstances) {
            node.model.meshInstances.forEach(function(mi) {
                if (mi.material === self.targetMaterial.resource) {
                    mi.material = self.shaderMaterial;
                }
            });
        } else if (node.render && node.render.meshInstances) {
            node.render.meshInstances.forEach(function(mi) {
                if (mi.material === self.targetMaterial.resource) {
                    mi.material = self.shaderMaterial;
                }
            });
        }
    });
};

shaderFallingStripes.prototype.update = function(dt) {
    // Mise à jour du temps
    this.time += dt;
    this.shaderMaterial.setParameter("time", this.time);

    // Mise à jour de la résolution (pour gérer les redimensionnements éventuels)
    var device = this.app.graphicsDevice;
    this.shaderMaterial.setParameter("iResolution", [device.width, device.height]);

    this.shaderMaterial.update();
};
