// Nom du script: FishJump.js
var FishJump = pc.createScript('fishJump');

// --- Paramètres configurables dans l'inspecteur PlayCanvas ---
FishJump.attributes.add('forwardSpeed', {
    type: 'number',
    default: 2,
    title: 'Vitesse Avant',
    description: 'Vitesse de déplacement en sous l’eau et pendant le saut (axe Z).'
});

FishJump.attributes.add('underWaterDistance', {
    type: 'number',
    default: 5,
    title: 'Distance Sous l’Eau',
    description: 'Distance que le poisson parcourt sous l’eau avant de faire un nouveau saut.'
});

FishJump.attributes.add('jumpDuration', {
    type: 'number',
    default: 2,
    title: 'Durée du Saut (secondes)',
    description: 'Temps (en secondes) que dure un saut complet.'
});

FishJump.attributes.add('jumpHeight', {
    type: 'number',
    default: 2,
    title: 'Hauteur du Saut',
    description: 'Hauteur maximale du saut (en unités de la scène).'
});

FishJump.attributes.add('maxTiltAngle', {
    type: 'number',
    default: 40,
    title: 'Inclinaison Max (degrés)',
    description: 'Angle d’inclinaison maximum vers le haut ou vers le bas.'
});

// --- Constantes pour l'état du poisson ---
var STATE_UNDERWATER = 0;
var STATE_JUMPING = 1;

FishJump.prototype.initialize = function() {

  
    // Position locale de départ (pour référence, si vous voulez y revenir plus tard)
    this.startPosition = this.entity.getLocalPosition().clone();

    // État initial: le poisson commence sous l’eau
    this.state = STATE_UNDERWATER;

    // Distance parcourue sous l’eau depuis le dernier saut
    this.distanceUnderwaterSoFar = 0;

    // Timer pour la phase de saut
    this.jumpTimer = 0;
};

// Fonction utilitaire: interpoler linéairement entre a et b selon t (entre 0 et 1).
FishJump.prototype.lerp = function(a, b, t) {
    return a + (b - a) * pc.math.clamp(t, 0, 1);
};

// Calcule l'angle d'inclinaison en fonction d'un « temps normalisé » t sur [0..1]
FishJump.prototype.computeTiltAngle = function(t) {
    // On divise le saut en 3 segments:
    // 1) 0   -> 0.25 : 0      -> +maxTilt
    // 2) 0.25-> 0.75 : +maxTilt -> -maxTilt
    // 3) 0.75-> 1    : -maxTilt -> 0
    var angle = 0;

    if (t < 0.25) {
        // De 0 à 0.25 : interpolation de 0 vers +maxTiltAngle
        var localT = t / 0.25;
        angle = this.lerp(0, this.maxTiltAngle, localT);
    } else if (t < 0.75) {
        // De 0.25 à 0.75 : interpolation de +maxTiltAngle vers -maxTiltAngle
        var localT = (t - 0.25) / 0.5;
        angle = this.lerp(this.maxTiltAngle, -this.maxTiltAngle, localT);
    } else {
        // De 0.75 à 1 : interpolation de -maxTiltAngle vers 0
        var localT = (t - 0.75) / 0.25;
        angle = this.lerp(-this.maxTiltAngle, 0, localT);
    }

    return angle;
};

FishJump.prototype.update = function(dt) {
    var currentPos = this.entity.getLocalPosition();

    if (this.state === STATE_UNDERWATER) {
        // Le poisson avance à l’horizontale, rotation = 0
        // On ajoute la distance parcourue cette frame
        var distanceThisFrame = this.forwardSpeed * dt;
        this.distanceUnderwaterSoFar += distanceThisFrame;

        // Mise à jour de la position (sur l’axe Z, local)
        currentPos.z += distanceThisFrame;
        // Forcer la position en Y=0 (si on veut rester pile "sous l’eau")
        currentPos.y = 0;
        this.entity.setLocalPosition(currentPos);

        // Remettre la rotation à 0° sur l’axe X
        this.entity.setLocalEulerAngles(0, 0, 0);

        // Quand on a parcouru la distance sous l’eau, on lance un saut
        if (this.distanceUnderwaterSoFar >= this.underWaterDistance) {
            this.state = STATE_JUMPING;
            this.jumpTimer = 0;
        }

    } else if (this.state === STATE_JUMPING) {
        // Avancement du timer du saut
        this.jumpTimer += dt;

        // Calcul du facteur t normalisé (entre 0 et 1) dans la phase de saut
        var t = this.jumpTimer / this.jumpDuration;
        if (t > 1) {
            t = 1;
        }

        // Position verticale : on utilise une sinusoïde simple
        // y(t) = jumpHeight * sin(pi * t)
        // t=0 => y=0, t=0.5 => y=jumpHeight, t=1 => y=0
        var newY = this.jumpHeight * Math.sin(Math.PI * t);

        // Mise à jour de la position
        currentPos.z += this.forwardSpeed * dt;  // on continue d’avancer
        currentPos.y = newY;
        this.entity.setLocalPosition(currentPos);

        // Calcul de l'orientation (pitch autour de X) selon la phase du saut
        var tilt = this.computeTiltAngle(t);
        this.entity.setLocalEulerAngles(tilt, 0, 0);

        // Fin du saut ?
        if (this.jumpTimer >= this.jumpDuration) {
            // On revient sous l’eau, on repasse l’angle à 0
            // et on réinitialise la distance sous l’eau
            currentPos.y = 0;
            this.entity.setLocalPosition(currentPos);
            this.entity.setLocalEulerAngles(0, 0, 0);

            this.state = STATE_UNDERWATER;
            this.distanceUnderwaterSoFar = 0;
        }
    }
};
