var shaderColorful = pc.createScript('shaderColorful');

// Attribut pour sélectionner, dans l'éditeur, le matériau cible à remplacer
shaderColorful.attributes.add('targetMaterial', {
    type: 'asset',
    assetType: 'material',
    title: 'Target Material'
});

// Attribut pour définir le centre de l'effet (valeurs normalisées, [0.5, 0.5] = centre de l'écran)
shaderColorful.attributes.add('center', {
    type: 'vec2',
    default: [0.5, 0.5],
    title: 'Center'
});

shaderColorful.prototype.initialize = function() {
    var device = this.app.graphicsDevice;

    // Vérifie qu'un matériau cible a été assigné
    if (!this.targetMaterial || !this.targetMaterial.resource) {
        console.warn("shaderColorful: Pas de 'Target Material' assigné !");
        return;
    }
    
    // --- Vertex Shader ---
    var vs = `
attribute vec3 aPosition;
attribute vec2 aUv0;
uniform mat4 matrix_model;
uniform mat4 matrix_viewProjection;
varying vec2 vUv0;
void main(void) {
    vUv0 = aUv0;
    gl_Position = matrix_viewProjection * matrix_model * vec4(aPosition, 1.0);
}
`;

    // --- Fragment Shader ---
    // On ajoute un uniform "center" pour décaler l'origine de l'effet
    var fs = `
precision ${device.precision} float;
varying vec2 vUv0;

// Uniforms
uniform float utime;
uniform vec2 iResolution;
uniform vec2 center; // Centre de l'effet (valeurs normalisées)

void main(void) {
    float time = utime*0.5;
    // Calcul du centre en pixels
    vec2 centerPixel = iResolution.xy * center;
    // Coordonnée pixel (fragCoord) à partir des UV
    vec2 fragCoord = vUv0 * iResolution;
    // Décalage des coordonnées pour recentrer l'effet
    vec2 uv = (fragCoord - centerPixel) / iResolution.y;
    
    vec3 color = vec3(0.0);
    const float PI = 3.14159265;
    const float G  = sqrt(5.0)*0.5 + 0.5;
    
    vec2 ar = vec2(atan(uv.y, uv.x), length(uv));
    vec3 c1 = cos(2.0 * PI * time + 1.0/ar.y - ar.x*vec3(1.0,2.0,3.0));
    vec3 c2 = cos(1.0 * PI * time + G/ar.y*vec3(1.7,0.5,1.0) - ar.x*vec3(2.0,3.0,1.0));
    vec3 c3 = cos(2.0 * PI * (time + sin(ar.x*6.0 - time)*(cos(time)*0.5+0.5))
                 + ar.x +6.0/ar.y*vec3(0.3,0.5,0.8));
    vec3 c4 = cos((G * PI * time)*vec3(0.7,1.0,1.3) + 4.0/ar.y - ar.x*vec3(3.0,1.0,2.0));
    
    color += exp(-5.0 * (0.5 + 0.5*c1) * exp(-0.2 / ar.y));
    color += 0.075*exp(-4.0 * (sqrt(c2*c2+0.125)-0.25) * exp(-0.2 / ar.y));
    color += 0.025*exp(-4.0 * (0.5 + 0.5*c3) * exp(-0.2 / ar.y));
    color += 0.050*exp(-4.0 * (sqrt(c4*c4+0.25)-0.5) * exp(-0.2 / ar.y));
    color *= 1.025 - 0.15 * dot(uv, uv);
    
    // Correction gamma
    color = pow(color, vec3(1.0/2.2));
    
    gl_FragColor = vec4(color, 1.0);
}
`;

    // Création du shader avec les attributs requis
    var shaderDefinition = {
        attributes: {
            aPosition: pc.SEMANTIC_POSITION,
            aUv0: pc.SEMANTIC_TEXCOORD0
        },
        vshader: vs,
        fshader: fs
    };
    this.shader = new pc.Shader(device, shaderDefinition);

    // Création du matériau custom et assignation du shader
    this.shaderMaterial = new pc.Material();
    this.shaderMaterial.shader = this.shader;
    this.shaderMaterial.cull = pc.CULLFACE_NONE; // Optionnel : désactive le face culling
    this.shaderMaterial.update();

    // Initialisation des uniforms
    this.time = 0;
    this.shaderMaterial.setParameter("utime", 0.0);
    this.shaderMaterial.setParameter("iResolution", [device.width, device.height]);
    this.shaderMaterial.setParameter("center", [this.center.x, this.center.y]);
    this.shaderMaterial.update();

    // Parcours de la hiérarchie pour remplacer le matériau cible par le shader custom
    var self = this;
    this.app.root.find(function(node) {
        if (node.model && node.model.meshInstances) {
            node.model.meshInstances.forEach(function(mi) {
                if (mi.material === self.targetMaterial.resource) {
                    mi.material = self.shaderMaterial;
                }
            });
        } else if (node.render && node.render.meshInstances) {
            node.render.meshInstances.forEach(function(mi) {
                if (mi.material === self.targetMaterial.resource) {
                    mi.material = self.shaderMaterial;
                }
            });
        }
    });
};

shaderColorful.prototype.update = function(dt) {
    this.time += dt;
    this.shaderMaterial.setParameter("utime", this.time);

    var device = this.app.graphicsDevice;
    this.shaderMaterial.setParameter("iResolution", [device.width, device.height]);
    this.shaderMaterial.setParameter("center", [this.center.x, this.center.y]);

    this.shaderMaterial.update();
};
