var GrowEntities = pc.createScript('growEntities');

// ─────────────────────────────────────────────────────────────────────────────────
// ──────────── PARAMÈTRES DU SCRIPT ─────────────────────────────────────────────
// ─────────────────────────────────────────────────────────────────────────────────

GrowEntities.attributes.add('eventName', {
    type: 'string',
    default: '',
    title: 'Event Name',
    description: 'Nom de l’événement déclencheur (vide => lancement immédiat)'
});

// Le "sol" : on récupère le bounding box pour déterminer la zone de spawn
GrowEntities.attributes.add('groundEntity', {
    type: 'entity',
    title: 'Ground Entity',
    description: 'Entité du sol (Render ou Model) pour déterminer la zone XZ'
});

// Zone d’exclusion (Render ou Model)
GrowEntities.attributes.add('exclusionZone', {
    type: 'entity',
    title: 'Exclusion Zone',
    description: 'Entité d’exclusion (optionnel), on écarte son bounding box'
});

/**
 * Liste d'entités à “popper”.
 *   - entityToSpawn : entité ou préfab
 *   - density       : combien d’instances au total
 *   - duration      : temps (s) sur lequel répartir la création (0 => instantané)
 *   - minScale / maxScale : pour randomiser l’échelle finale
 */
GrowEntities.attributes.add('spawnedEntities', {
    type: 'json',
    array: true,
    schema: [
        {
            name: 'entityToSpawn',
            type: 'entity',
            title: 'Entity to Spawn'
        },
        {
            name: 'density',
            type: 'number',
            default: 5,
            title: 'Density'
        },
        {
            name: 'duration',
            type: 'number',
            default: 0,
            title: 'Duration (sec)',
            description: 'Temps pour répartir la création (0 => spawn instantané)'
        },
        {
            name: 'minScale',
            type: 'vec3',
            default: [0.1, 0.1, 0.1],
            title: 'Min Scale'
        },
        {
            name: 'maxScale',
            type: 'vec3',
            default: [1, 1, 1],
            title: 'Max Scale'
        }
    ],
    title: 'Spawned Entities'
});

// Vitesse d’apparition (multiplicateur de vitesse pour l’effet “pop”)
GrowEntities.attributes.add('appearSpeed', {
    type: 'number',
    default: 1,
    title: 'Appear Speed'
});

// Si on veut une rotation Y aléatoire
GrowEntities.attributes.add('randomRotation', {
    type: 'boolean',
    default: true,
    title: 'Random Rotation'
});

// Intensité du rebond (“pop cartoon”)
GrowEntities.attributes.add('bounceStrength', {
    type: 'number',
    default: 1.70158,
    title: 'Bounce Strength',
    description: 'Facteur de rebond pour l’effet “pop” (easeOutBack)'
});

// ─────────────────────────────────────────────────────────────────────────────────
// ──────────── INITIALISATION ───────────────────────────────────────────────────
// ─────────────────────────────────────────────────────────────────────────────────

GrowEntities.prototype.initialize = function() {
    // Liste d’entités en croissance
    this._growingEntities = [];

    // Liste des “queues” pour un spawn progressif
    this._spawnQueues = [];

    // On déclenche tout de suite ou on attend l’event
    if (this.eventName && this.eventName.length > 0) {
        this.app.on(this.eventName, this.onSpawnTriggered, this);
    } else {
        this.onSpawnTriggered();
    }
};

// ─────────────────────────────────────────────────────────────────────────────────
// ──────────── LANCEMENT DU SPAWN ───────────────────────────────────────────────
// ─────────────────────────────────────────────────────────────────────────────────

GrowEntities.prototype.onSpawnTriggered = function() {

    // Log
    if (this.eventName && this.eventName.length > 0) {
        console.log(this.eventName + ' : ok');
    } else {

        console.log('Default spawn triggered : ok');
    }

    // Pour chaque config, on échelonne ou on spawn direct
    for (var i = 0; i < this.spawnedEntities.length; i++) {
        var cfg = this.spawnedEntities[i];
        var total = Math.max(0, cfg.density || 0);

        if (cfg.duration > 0) {
            this._spawnQueues.push({
                config: cfg,
                total: total,
                duration: cfg.duration,
                elapsed: 0,
                spawnedCount: 0
            });
        } else {
            this.spawnAllAtOnce(cfg, total);
        }
    }
};

// Spawn “en bloc”
GrowEntities.prototype.spawnAllAtOnce = function(config, count) {
    for (var c = 0; c < count; c++) {
        var instance = this.spawnOneEntity(config);
        if (instance) {
            this._growingEntities.push({
                entity: instance,
                targetScale: instance.targetScale.clone(),
                elapsed: 0
            });
            instance.setLocalScale(0, 0, 0); // part de scale 0
        }
    }
};

// ─────────────────────────────────────────────────────────────────────────────────
// ──────────── UPDATE ───────────────────────────────────────────────────────────
// ─────────────────────────────────────────────────────────────────────────────────

GrowEntities.prototype.update = function(dt) {
    this.updateSpawnQueues(dt);
    this.updateGrowingEntities(dt);
};

// ─────────────────────────────────────────────────────────────────────────────────
// ──────────── SPAWN PROGRESSIF ─────────────────────────────────────────────────
// ─────────────────────────────────────────────────────────────────────────────────

GrowEntities.prototype.updateSpawnQueues = function(dt) {
    for (var i = this._spawnQueues.length - 1; i >= 0; i--) {
        var queue = this._spawnQueues[i];
        queue.elapsed += dt;

        var rate = queue.total / queue.duration; // entités / sec
        var expectedSoFar = rate * queue.elapsed;
        var toSpawn = Math.floor(expectedSoFar - queue.spawnedCount);

        // Éviter de dépasser le total
        if (queue.spawnedCount + toSpawn > queue.total) {
            toSpawn = queue.total - queue.spawnedCount;
        }

        for (var s = 0; s < toSpawn; s++) {
            var instance = this.spawnOneEntity(queue.config);
            if (instance) {
                this._growingEntities.push({
                    entity: instance,
                    targetScale: instance.targetScale.clone(),
                    elapsed: 0
                });
                instance.setLocalScale(0, 0, 0);
            }
            queue.spawnedCount++;
        }

        // Si on a atteint la fin
        if (queue.spawnedCount >= queue.total) {
            this._spawnQueues.splice(i, 1);
        }
    }
};

// ─────────────────────────────────────────────────────────────────────────────────
// ──────────── ANIMATION “POP” DES ENTITÉS ──────────────────────────────────────
// ─────────────────────────────────────────────────────────────────────────────────

GrowEntities.prototype.updateGrowingEntities = function(dt) {
    for (var i = this._growingEntities.length - 1; i >= 0; i--) {
        var info = this._growingEntities[i];
        info.elapsed += dt;

        // t = fraction du temps écoulé (0 => 1)
        var t = pc.math.clamp(info.elapsed * this.appearSpeed, 0, 1);

        // On applique un easing “easeOutBack” pour avoir un pop
        var popFactor = this.easeOutBack(t);

        // On multiplie la targetScale par ce popFactor
        var currentScale = new pc.Vec3(
            info.targetScale.x * popFactor,
            info.targetScale.y * popFactor,
            info.targetScale.z * popFactor
        );
        info.entity.setLocalScale(currentScale);

        // Si complètement apparu
        if (t >= 1) {
            this._growingEntities.splice(i, 1);
        }
    }
};

// Fonction d’easing “easeOutBack” (selon Penner)
GrowEntities.prototype.easeOutBack = function(t) {
    // Contrôle la quantité de “rebond”
    var s = this.bounceStrength;

    // Formule standard : t’ = (t - 1)
    t = t - 1;
    return 1 + t * t * ((s + 1) * t + s);
};

// ─────────────────────────────────────────────────────────────────────────────────
// ──────────── SPAWN D’UNE ENTITÉ UNIQUE ─────────────────────────────────────────
// ─────────────────────────────────────────────────────────────────────────────────

GrowEntities.prototype.spawnOneEntity = function(config) {
    if (!config.entityToSpawn) return null;
    if (!this.groundEntity) return null;

    var newEntity = config.entityToSpawn.clone();
    newEntity.enabled = true;

    // 1) Récupération du bounding box du sol
    var groundAabb = this.getCombinedAabbFromEntity(this.groundEntity);
    if (!groundAabb) {
        console.warn('No bounding box found for groundEntity. Aborting spawn.');
        return null;
    }
    // On sync avant lecture
    // (déjà fait dans getCombinedAabbFromEntity, mais on le refait “au cas où”)
    // => normal : la fonction s’occupe de sync déjà

    // 2) On génère une position XZ aléatoire à l’intérieur du bounding box
    var randomPos = this.getRandomPositionInAabb(groundAabb);

    // 3) On vérifie la zone d’exclusion (30 essais)
    var maxTry = 30;
    for (var t = 0; t < maxTry; t++) {
        if (!this.isInsideExclusionZone(randomPos)) {
            break;
        }
        randomPos = this.getRandomPositionInAabb(groundAabb);
    }

    // 4) Y : on prend par exemple la position “maxY” du bounding box (pour être au-dessus)
    //    ou la position center. Ici, on va prendre “maxY” pour un plane standard.
    var y = groundAabb.getMax().y;

    // On place l’entité
    newEntity.setPosition(randomPos.x, y, randomPos.z);

    // 5) Scale final (aléatoire)
    var finalScale = this.getRandomScale(config.minScale, config.maxScale);
    newEntity.targetScale = finalScale;

    // 6) Rotation aléatoire
    if (this.randomRotation) {
        newEntity.setLocalEulerAngles(0, Math.random() * 360, 0);
    }

    // 7) Ajout à la scène
    this.app.root.addChild(newEntity);
    return newEntity;
};

// ─────────────────────────────────────────────────────────────────────────────────
// ──────────── OUTILS (Bounding Box / Random / Exclusion) ───────────────────────
// ─────────────────────────────────────────────────────────────────────────────────

/**
 * Récupère le bounding box mondial (AABB) “global” d’une entité
 * qui dispose d’un RenderComponent ou ModelComponent.
 * Combine toutes les meshInstances si besoin.
 * Retourne un pc.BoundingBox ou null si rien n’est trouvé.
 */
GrowEntities.prototype.getCombinedAabbFromEntity = function(entity) {
    var meshInstances = null;

    // RenderComponent (nouveau)
    if (entity.render && entity.render.meshInstances) {
        meshInstances = entity.render.meshInstances;
    }
    // ModelComponent (ancien)
    else if (entity.model && entity.model.meshInstances) {
        meshInstances = entity.model.meshInstances;
    }

    if (!meshInstances || meshInstances.length === 0) {
        return null;
    }

    // On combine tout
    var combinedAabb = null;
    for (var i = 0; i < meshInstances.length; i++) {
        var mi = meshInstances[i];
        mi.syncAabb(); // Mise à jour
        if (!combinedAabb) {
            combinedAabb = mi.aabb.clone();
        } else {
            combinedAabb.add(mi.aabb);
        }
    }
    return combinedAabb;
};

/**
 * Renvoie une position (x, 0, z) aléatoire
 * à l’intérieur d’un AABB donné (en XZ).
 * Y n’est pas utilisé ici => on mettra Y plus tard.
 */
GrowEntities.prototype.getRandomPositionInAabb = function(aabb) {
    var min = aabb.getMin();
    var max = aabb.getMax();

    // X dans [minX, maxX], Z dans [minZ, maxZ]
    var x = pc.math.random(min.x, max.x);
    var z = pc.math.random(min.z, max.z);

    return new pc.Vec3(x, 0, z);
};

/**
 * Renvoie un pc.Vec3 aléatoire entre minScale et maxScale.
 */
GrowEntities.prototype.getRandomScale = function(minScale, maxScale) {
    var x = pc.math.random(minScale.x, maxScale.x);
    var y = pc.math.random(minScale.y, maxScale.y);
    var z = pc.math.random(minScale.z, maxScale.z);
    return new pc.Vec3(x, y, z);
};

/**
 * Vérifie si posXZ (pc.Vec3) se trouve dans la zone d’exclusion.
 * (On se base sur le bounding box complet de l’exclusionZone.)
 */
GrowEntities.prototype.isInsideExclusionZone = function(posXZ) {
    if (!this.exclusionZone) {
        return false;
    }

    var exclusionAabb = this.getCombinedAabbFromEntity(this.exclusionZone);
    if (!exclusionAabb) {
        return false;
    }

    var minX = exclusionAabb.getMin().x;
    var minZ = exclusionAabb.getMin().z;
    var maxX = exclusionAabb.getMax().x;
    var maxZ = exclusionAabb.getMax().z;

    return (
        posXZ.x >= minX &&
        posXZ.x <= maxX &&
        posXZ.z >= minZ &&
        posXZ.z <= maxZ
    );
};
