var ExplosionCone = pc.createScript('explosionCone');

// 🎯 Nom de l'événement à écouter
ExplosionCone.attributes.add('triggerEventName', {
    type: 'string',
    default: 'explosion',
    title: "Nom de l'événement déclencheur"
});

// ⚡ Angle d'explosion (en degrés)
ExplosionCone.attributes.add('explosionAngle', {
    type: 'number',
    default: 45,
    title: "Angle d'explosion"
});

// ↔️ Distance maximale parcourue
ExplosionCone.attributes.add('maxDistance', {
    type: 'number',
    default: 10,
    title: "Distance maximale parcourue"
});

// ⏳ Durée totale de l'animation (sec)
ExplosionCone.attributes.add('duration', {
    type: 'number',
    default: 2,
    title: "Durée de l'animation"
});

// 💨 Vitesse d'expansion
ExplosionCone.attributes.add('expansionSpeed', {
    type: 'number',
    default: 1,
    title: "Vitesse d'expansion"
});

// 🌈 Type d'interpolation
ExplosionCone.attributes.add('interpolationType', {
    type: 'string',
    enum: [
        { "Linaire": "linear" },
        { "Exponentielle": "exponential" }
    ],
    default: "linear",
    title: "Type d'interpolation"
});

// ↘️ Effet de ralentissement (true = décélération en fin de mouvement)
ExplosionCone.attributes.add('slowdownEffect', {
    type: 'boolean',
    default: true,
    title: "Effet de ralentissement"
});

// 🌀 Entités à dupliquer (jusqu'à 3 entités)
ExplosionCone.attributes.add('entity1', { type: 'entity', title: "Entité 1" });
ExplosionCone.attributes.add('entity2', { type: 'entity', title: "Entité 2" });
ExplosionCone.attributes.add('entity3', { type: 'entity', title: "Entité 3" });

// 🌟 Nombre d'instances par entité
ExplosionCone.attributes.add('numInstances', {
    type: 'number',
    default: 5,
    title: "Nombre d'instances par entité"
});

// ⚡ Taux d'émission (en secondes entre chaque cycle d'émission)
ExplosionCone.attributes.add('emissionRate', {
    type: 'number',
    default: 0.1,
    title: "Taux d'émission"
});

////////////////////////////////////////////////////////////////////////////////
// ░░░░░ INITIALISATION ░░░░░
////////////////////////////////////////////////////////////////////////////////

ExplosionCone.prototype.initialize = function() {
    this._elapsedTime = 0;
    this._animating = false;

    // La file d'attente des particules instanciées
    this._particleQueue = [];

    // Quand l'événement spécifié est déclenché, on lance startExplosion
    if (this.triggerEventName && this.triggerEventName !== "") {
        this.app.on(this.triggerEventName, this.startExplosion, this);
    }
};

////////////////////////////////////////////////////////////////////////////////
// ░░░░░ GENERATION / EMISSION DES ENTITES ░░░░░
////////////////////////////////////////////////////////////////////////////////

ExplosionCone.prototype.spawnEntities = function() {
    // Récupérer les entités "template" non nulles
    var entityTemplates = [this.entity1, this.entity2, this.entity3].filter(e => e);

    // Pour chaque template, cloner 'numInstances' fois
    for (var i = 0; i < this.numInstances; i++) {
        entityTemplates.forEach(template => {
            var instance = template.clone();
            // Ajouter l'instance dans la hiérarchie (ex: même parent que ce script)
            this.entity.getParent().addChild(instance);

            // Position de départ = position de l'entité qui porte ce script
            instance.setPosition(this.entity.getPosition());

            // On initialise éventuellement l'échelle à 0.01 si on veut partir très petit
            instance.setLocalScale(0.01, 0.01, 0.01);

            // On stocke cette instance dans la queue
            this._particleQueue.push(instance);
        });
    }
};

////////////////////////////////////////////////////////////////////////////////
// ░░░░░ START EXPLOSION ░░░░░
// => On crée les entités à émettre, on enclenche l'animation
////////////////////////////////////////////////////////////////////////////////

ExplosionCone.prototype.startExplosion = function() {
    // Générer d'abord toutes les entités instanciées
    this.spawnEntities();

    // Reset le temps et démarre l'animation
    this._elapsedTime = 0;
    this._animating = true;
};

////////////////////////////////////////////////////////////////////////////////
// ░░░░░ UPDATE (ANIMATION) ░░░░░
////////////////////////////////////////////////////////////////////////////////

ExplosionCone.prototype.update = function(dt) {
    if (!this._animating) return;

    this._elapsedTime += dt;
    // Calculer le ratio t = [0..1]
    var t = this._elapsedTime / this.duration;
    if (t > 1) t = 1;

    // Choisir la fonction d'interpolation
    var progress;
    if (this.interpolationType === "linear") {
        progress = this.slowdownEffect ? this.easeOutQuad(t) : t;
    } else {
        // exponentiel
        progress = this.slowdownEffect ? this.easeOutExpo(t) : t;
    }

    // Parcourir tous les clones instanciés
    for (var i = 0; i < this._particleQueue.length; i++) {
        var entity = this._particleQueue[i];

        // Générer un vecteur direction aléatoire dans le cône
        var direction = this._getRandomConeDirection(this.explosionAngle);
        direction.normalize();
        direction.scale(this.maxDistance);

        // Position finale
        var targetPos = this.entity.getPosition().clone().add(direction);

        // On fait un lerp entre la position de départ (this.entity) et targetPos
        var currentPos = new pc.Vec3();
        currentPos.lerp(this.entity.getPosition(), targetPos, progress);
        entity.setPosition(currentPos);

        // Echelle: on part de 0.01, on va vers l'échelle finale (1,1,1) ou ajustée
        // On peut multiplier par expansionSpeed * t pour arriver à 1 + ...
        var scaleFactor = this.expansionSpeed * t;
        if (scaleFactor > 1) {
            scaleFactor = 1;
        }
        entity.setLocalScale(scaleFactor, scaleFactor, scaleFactor);
    }

    // Si l'animation est terminée, on arrête
    if (t >= 1) {
        this._animating = false;
    }
};

////////////////////////////////////////////////////////////////////////////////
// ░░░░░ FONCTIONS UTILES ░░░░░
////////////////////////////////////////////////////////////////////////////////

// Obtenir une direction aléatoire dans un cône d'ouverture 'angleDegrees'
ExplosionCone.prototype._getRandomConeDirection = function(angleDegrees) {
    // Convertir l'angle en radians
    var coneAngleRad = angleDegrees * pc.math.DEG_TO_RAD;

    // theta = angle azimutal (autour de Y), phi = angle par rapport à l'axe Y
    var theta = Math.random() * 2 * Math.PI;
    var phi = Math.random() * coneAngleRad; // angle inclus dans le cône

    // Calcul en coordonnées sphériques, axe Y = up
    // phi=0 => pointe vers le haut, phi=coneAngleRad => angle max
    var sinPhi = Math.sin(phi);
    var x = sinPhi * Math.cos(theta);
    var y = Math.cos(phi);  // orienté vers le haut
    var z = sinPhi * Math.sin(theta);

    return new pc.Vec3(x, y, z);
};

// EaseOutQuad
ExplosionCone.prototype.easeOutQuad = function(t) {
    return t * (2 - t);
};

// EaseOutExpo
ExplosionCone.prototype.easeOutExpo = function(t) {
    return (t === 1) ? 1 : 1 - Math.pow(2, -10 * t);
};
