var OrbLoadingbar = pc.createScript('orbLoadingbar');

/**
 * Attributs configurables
 */
OrbLoadingbar.attributes.add('targetEntity', {
    type: 'entity',
    title: 'Entité à animer',
    description: 'Drag & drop l’entité (ex: cube, sphère) que vous voulez animer.'
});

OrbLoadingbar.attributes.add('animationDuration', {
    type: 'number',
    default: 1,
    title: 'Durée de l’animation (sec)',
    description: 'Temps en secondes pour l\'animation (plus c\'est grand, plus l’animation est lente).'
});

OrbLoadingbar.attributes.add('startDelayMs', {
    type: 'number',
    default: 0,
    title: 'Délai avant anim (ms)',
    description: 'Temps en millisecondes à patienter avant de commencer l’animation.'
});

// --- NOUVEAUX attributs ---
OrbLoadingbar.attributes.add('endPosition', {
    type: 'vec3',
    default: [2.1814935207366943, 58.548828125, 12.100506782531738],
    title: 'Position Finale',
    description: 'Position (Monde ou Locale) que l\'entité doit atteindre.'
});

// Direction finale utilisée pour orienter l’entité
OrbLoadingbar.attributes.add('endDirection', {
    type: 'vec3',
    default: [-0.02403315754794052, 0.984195606114128, -0.1754463343701436],
    title: 'Direction Finale',
    description: 'Direction “forward” pour l’entité, une fois arrivée à destination.'
});

// Paramètres pour l’effet de pulsation (coeur)
OrbLoadingbar.attributes.add('pulseSpeed', {
    type: 'number',
    default: 2,
    title: 'Vitesse pulsation'
});

OrbLoadingbar.attributes.add('pulseAmplitude', {
    type: 'number',
    default: 0.1,
    title: 'Amplitude pulsation'
});

// initialize : s'exécute une fois au démarrage
OrbLoadingbar.prototype.initialize = function() {
    // 1) Met l’entité à l’échelle 0,0,0 pour la « cacher »
    this.targetEntity.setLocalScale(0, 0, 0);

    // 2) Stocke la position de départ (locale)
    this.startPos = this.targetEntity.getLocalPosition().clone();

    // 3) Variables pour gérer l’animation de déplacement
    this.isAnimating = true;
    this.elapsed = 0;

    // 4) Variables pour gérer le délai
    this.waiting = false;
    this.waitTimer = 0;

    // 5) Variable pour l’effet de pulsation
    this.pulseActive = false;    // s'activera quand l’animation de déplacement sera terminée
    this.pulseTime = 0;         // pour faire progresser la sinusoïde

    // 6) On écoute l’événement "barcamAnim"
    this.app.on('barcamAnim', function() {
        // Réinitialise l’échelle et la position
        this.targetEntity.setLocalScale(0, 0, 0);
        this.targetEntity.setLocalPosition(this.startPos);

        // Réinitialise les compteurs
        this.elapsed = 0;
        this.waitTimer = 0;
        this.isAnimating = false;
        this.waiting = false;
        this.pulseActive = false;
        this.pulseTime = 0;

        // Si un délai est spécifié, on attend avant de démarrer l’animation
        if (this.startDelayMs > 0) {
            this.waiting = true;
        } else {
            // Sinon, on lance l’animation immédiatement
            this.isAnimating = true;
        }
    }, this);
};

// update : s'exécute à chaque frame
OrbLoadingbar.prototype.update = function(dt) {
    // 1) Gérer le délai (startDelayMs)
    if (this.waiting) {
        this.waitTimer += dt * 1000; // dt est en secondes, on convertit en ms
        if (this.waitTimer >= this.startDelayMs) {
            // Fin du délai, on lance l’animation
            this.waiting = false;
            this.isAnimating = true;
        }
    }

    // 2) Gérer l’animation de déplacement
    if (this.isAnimating) {
        this.elapsed += dt;

        // ratio entre 0 et 1
        var ratio = pc.math.clamp(this.elapsed / this.animationDuration, 0, 1);

        // a) Faire grandir l’échelle de (0 -> 1)
        var newScale = pc.math.lerp(0, 1, ratio);
        this.targetEntity.setLocalScale(newScale, newScale, newScale);

        // b) Faire varier la position de startPos -> endPosition
        var currentPos = new pc.Vec3();
        currentPos.x = pc.math.lerp(this.startPos.x, this.endPosition.x, ratio);
        currentPos.y = pc.math.lerp(this.startPos.y, this.endPosition.y, ratio);
        currentPos.z = pc.math.lerp(this.startPos.z, this.endPosition.z, ratio);
        this.targetEntity.setLocalPosition(currentPos);

        // c) Si on a atteint la fin (ratio >= 1), on arrête l’animation et on place l’entité exactement
        if (ratio >= 1) {
            this.isAnimating = false;

            // On s’assure que la position est bien celle voulue
            this.targetEntity.setLocalPosition(this.endPosition);

            // On oriente l’entité selon la direction souhaitée
            var lookAtPos = this.endPosition.clone().add(this.endDirection);
            this.targetEntity.lookAt(lookAtPos);

            // On active la pulsation "cœur"
            this.pulseActive = true;
        }
    }

    // 3) Gérer l’effet de pulsation (coeur) une fois l’animation terminée
    if (this.pulseActive) {
        this.pulseTime += dt;
        // facteur de pulsation sinusoïdale
        var pulsFactor = 1 + Math.sin(this.pulseTime * this.pulseSpeed) * this.pulseAmplitude;
        this.targetEntity.setLocalScale(pulsFactor, pulsFactor, pulsFactor);
    }
};
