var MeshSlicerCenter = pc.createScript('meshSlicerCenter');

// On garde les mêmes attributs qu'avant, sauf qu'on VA calculer planePoint !
MeshSlicerCenter.attributes.add('targetEntity', {
    type: 'entity',
    title: 'Target Entity',
    description: 'Entité dont on veut slicer le maillage'
});

// On peut toujours garder un champ pour la planeNormal
// ou on la fixe en code si on veut.
MeshSlicerCenter.attributes.add('planeNormal', {
    type: 'vec3',
    default: [0, 1, 0],
    title: 'Plane Normal (Local)',
    description: 'La normale (locale) du plan'
});

MeshSlicerCenter.attributes.add('explodeForce', {
    type: 'number',
    default: 0.2,
    title: 'Explode Force'
});

MeshSlicerCenter.prototype.initialize = function() {
    if (!this.targetEntity || !this.targetEntity.model) {
        console.warn("MeshSlicerCenter: targetEntity invalide ou sans ModelComponent");
        return;
    }
    var meshInstances = this.targetEntity.model.meshInstances;
    if (!meshInstances || meshInstances.length === 0) {
        console.warn("MeshSlicerCenter: Pas de meshInstances sur targetEntity");
        return;
    }

    var mesh = meshInstances[0].mesh;
    if (!mesh) {
        console.warn("MeshSlicerCenter: mesh introuvable");
        return;
    }

    // 1) Extraire les données (positions + indices) pour calculer un bounding box LOCAL
    var meshData = this._extractMeshData(mesh);
    if (!meshData) return;

    // 2) Calculer le bounding box local via la même fonction `_computeAABB`
    var localAabb = this._computeAABB(meshData.positions);
    // On récupère son centre local
    var localCenter = localAabb.center.clone();

    // => C'est ici qu'on dit : planePoint = le centre du bounding box
    // On *peut* le laisser en local, puisque tout le reste du code
    // va faire la découpe en local (comme dans l'exemple).
    // Donc on définit planePoint localement :
    var localPlanePoint = localCenter;

    // 3) Normale : on conserve planeNormal telle quelle, en supposant qu'on est en local.
    var localPlaneNormal = new pc.Vec3(this.planeNormal.x, this.planeNormal.y, this.planeNormal.z).normalize();

    // 4) Passer à la routine de slicing
    var sliceResult = this._sliceMesh(meshData, localPlanePoint, localPlaneNormal);

    // 5) Générer les deux meshes
    var meshPos = this._createPcMesh(sliceResult.posSide, this.app.graphicsDevice);
    var meshNeg = this._createPcMesh(sliceResult.negSide, this.app.graphicsDevice);

    if (!meshPos && !meshNeg) {
        console.warn("MeshSlicerCenter: La coupe ne génère aucune géométrie (peut-être hors du mesh ?)");
        return;
    }

    // On récupère le matériau d’origine
    var originalMat = meshInstances[0].material;

    // On crée deux entités pour accueillir les deux moitiés
    var parent = this.targetEntity.parent;

    var ePos = new pc.Entity("Sliced_Positive");
    ePos.addComponent("model", {
        type: "asset",
        castShadows: this.targetEntity.model.castShadows,
        receiveShadows: this.targetEntity.model.receiveShadows
    });
    parent.addChild(ePos);

    var eNeg = new pc.Entity("Sliced_Negative");
    eNeg.addComponent("model", {
        type: "asset",
        castShadows: this.targetEntity.model.castShadows,
        receiveShadows: this.targetEntity.model.receiveShadows
    });
    parent.addChild(eNeg);

    // Copier la transf locale
    ePos.setLocalPosition(this.targetEntity.getLocalPosition());
    ePos.setLocalRotation(this.targetEntity.getLocalRotation());
    ePos.setLocalScale(this.targetEntity.getLocalScale());

    eNeg.setLocalPosition(this.targetEntity.getLocalPosition());
    eNeg.setLocalRotation(this.targetEntity.getLocalRotation());
    eNeg.setLocalScale(this.targetEntity.getLocalScale());

    // Assigner les meshes et le material
    if (meshPos) {
        ePos.model.model.meshInstances[0].mesh = meshPos;
        ePos.model.model.meshInstances[0].material = originalMat;
    } else {
        ePos.enabled = false;
    }

    if (meshNeg) {
        eNeg.model.model.meshInstances[0].mesh = meshNeg;
        eNeg.model.model.meshInstances[0].material = originalMat;
    } else {
        eNeg.enabled = false;
    }

    // Désactiver l'entité d'origine
    this.targetEntity.enabled = false;

    // Appliquer un léger offset pour simuler l'explosion
    if (ePos.enabled) {
        var offsetPos = localPlaneNormal.clone().scale(this.explodeForce);
        ePos.translate(offsetPos);
    }
    if (eNeg.enabled) {
        var offsetNeg = localPlaneNormal.clone().scale(-this.explodeForce);
        eNeg.translate(offsetNeg);
    }

    console.log("MeshSlicerCenter: Slicing au milieu + explosion terminé !");
};


// === Les fonctions _extractMeshData, _sliceMesh, _createPcMesh, _computeAABB
//     sont exactement les mêmes que dans l'exemple précédent, je ne les recolle pas ici 
//     par souci de concision, mais vous devez les inclure à l'identique (ou les importer).

/**
 * _extractMeshData(mesh)
 * Récupère les positions et indices du mesh.
 * (Ne gère pas dans cet exemple les normales ni les UV)
 */
MeshSlicerCenter.prototype._extractMeshData = function(mesh) {
    var vertexBuffer = mesh.vertexBuffer;
    if (!vertexBuffer) {
        console.error("MeshSlicerCenter: Pas de vertexBuffer");
        return null;
    }
    var indexBuffer = mesh.indexBuffer[0];
    if (!indexBuffer) {
        console.error("MeshSlicerCenter: Pas de indexBuffer[0]");
        return null;
    }
    var format = vertexBuffer.getFormat();

    // Chercher la sémantique POSITION
    var positionSem = null;
    for (var i = 0; i < format.elements.length; i++) {
        var element = format.elements[i];
        if (element.name === pc.SEMANTIC_POSITION) {
            positionSem = element;
            break;
        }
    }
    if (!positionSem) {
        console.error("MeshSlicerCenter: Impossible de trouver SEMANTIC_POSITION");
        return null;
    }

    // Extraire les positions
    var iterator = new pc.VertexIterator(vertexBuffer);
    var positions = new Array(vertexBuffer.numVertices);
    
    for (var v = 0; v < vertexBuffer.numVertices; v++) {
        var x = iterator.element[pc.SEMANTIC_POSITION].x;
        var y = iterator.element[pc.SEMANTIC_POSITION].y;
        var z = iterator.element[pc.SEMANTIC_POSITION].z;
        
        positions[v] = new pc.Vec3(x, y, z);
        
        iterator.next();
    }
    iterator.end();

    // Extraire les indices
    var indices = new Uint16Array(indexBuffer.lock());

    return {
        positions: positions,
        indices: indices
    };
};


/**
 * _sliceMesh(meshData, planePt, planeNormal)
 * Découpe la géométrie en deux séries de triangles
 * (côté positif et côté négatif du plan).
 */
MeshSlicerCenter.prototype._sliceMesh = function(meshData, planePt, planeNormal) {
    var posSide = {
        positions: [],
        indices: []
    };
    var negSide = {
        positions: [],
        indices: []
    };

    var positions = meshData.positions;
    var indices = meshData.indices;

    var signedDist = function(p) {
        // distance = dot((p - planePt), planeNormal)
        var v = p.clone().sub(planePt);
        return v.dot(planeNormal);
    };

    var processTriangle = (i0, i1, i2) => {
        var p0 = positions[i0];
        var p1 = positions[i1];
        var p2 = positions[i2];

        var d0 = signedDist(p0);
        var d1 = signedDist(p1);
        var d2 = signedDist(p2);

        var side0 = d0 >= 0 ? 1 : -1;
        var side1 = d1 >= 0 ? 1 : -1;
        var side2 = d2 >= 0 ? 1 : -1;

        // Tout le triangle d’un côté
        if (side0 === side1 && side1 === side2) {
            if (side0 > 0) {
                var baseIndex = posSide.positions.length;
                posSide.positions.push(p0.clone(), p1.clone(), p2.clone());
                posSide.indices.push(baseIndex, baseIndex+1, baseIndex+2);
            } else {
                var baseIndex = negSide.positions.length;
                negSide.positions.push(p0.clone(), p1.clone(), p2.clone());
                negSide.indices.push(baseIndex, baseIndex+1, baseIndex+2);
            }
        } else {
            // Le triangle est coupé par le plan 
            // => calcul d’intersections
            var allP = [p0, p1, p2];
            var allD = [d0, d1, d2];
            var side = [side0, side1, side2];

            var posVerts = [];
            var negVerts = [];

            for (var e = 0; e < 3; e++) {
                var e0 = e;
                var e1 = (e+1) % 3;

                var pA = allP[e0];
                var pB = allP[e1];
                var dA = allD[e0];
                var dB = allD[e1];
                var sA = side[e0];
                var sB = side[e1];

                if (sA >= 0) posVerts.push(pA);
                else negVerts.push(pA);

                if (sA * sB < 0) {
                    // On coupe l'edge
                    var t = Math.abs(dA) / (Math.abs(dA) + Math.abs(dB));
                    var intersect = pA.clone().lerp(pB, t);
                    posVerts.push(intersect);
                    negVerts.push(intersect.clone());
                }
            }

            // Triangulation simple
            if (posVerts.length >= 3) {
                var baseIndex = posSide.positions.length;
                posSide.positions.push(posVerts[0], posVerts[1], posVerts[2]);
                posSide.indices.push(baseIndex, baseIndex+1, baseIndex+2);
                if (posVerts[3]) {
                    // second triangle
                    var b2 = posSide.positions.length;
                    posSide.positions.push(posVerts[0], posVerts[2], posVerts[3]);
                    posSide.indices.push(b2, b2+1, b2+2);
                }
            }
            if (negVerts.length >= 3) {
                var baseIndex = negSide.positions.length;
                negSide.positions.push(negVerts[0], negVerts[1], negVerts[2]);
                negSide.indices.push(baseIndex, baseIndex+1, baseIndex+2);
                if (negVerts[3]) {
                    var b2 = negSide.positions.length;
                    negSide.positions.push(negVerts[0], negVerts[2], negVerts[3]);
                    negSide.indices.push(b2, b2+1, b2+2);
                }
            }
        }
    };

    for (var t = 0; t < indices.length; t += 3) {
        var i0 = indices[t];
        var i1 = indices[t+1];
        var i2 = indices[t+2];
        processTriangle(i0, i1, i2);
    }

    return {
        posSide: posSide,
        negSide: negSide
    };
};


/**
 * _createPcMesh(meshData, device)
 * Construit un pc.Mesh à partir d'un tableau de positions + indices.
 */
MeshSlicerCenter.prototype._createPcMesh = function(meshData, device) {
    if (!meshData.positions || meshData.positions.length === 0) {
        return null;
    }

    var vertexFormat = new pc.VertexFormat(device, [
        { semantic: pc.SEMANTIC_POSITION, components: 3, type: pc.TYPE_FLOAT32 }
    ]);

    var numVerts = meshData.positions.length;
    var vertexBuffer = new pc.VertexBuffer(device, vertexFormat, numVerts);

    var iterator = new pc.VertexIterator(vertexBuffer);
    for (var i = 0; i < numVerts; i++) {
        var pos = meshData.positions[i];
        iterator.element[pc.SEMANTIC_POSITION].set(pos.x, pos.y, pos.z);
        iterator.next();
    }
    iterator.end();

    var indices = new Uint16Array(meshData.indices);
    var indexBuffer = new pc.IndexBuffer(device, pc.INDEXFORMAT_UINT16, indices.length);
    var dst = new Uint16Array(indexBuffer.lock());
    dst.set(indices);
    indexBuffer.unlock();

    var newMesh = new pc.Mesh();
    newMesh.vertexBuffer = vertexBuffer;
    newMesh.indexBuffer[0] = indexBuffer;
    newMesh.primitive[0].type = pc.PRIMITIVE_TRIANGLES;
    newMesh.primitive[0].base = 0;
    newMesh.primitive[0].count = indices.length;
    newMesh.primitive[0].indexed = true;

    // Calcul approximatif du bounding box
    newMesh.aabb = new pc.BoundingBox();
    newMesh.aabb.copy(this._computeAABB(meshData.positions));

    return newMesh;
};


/**
 * _computeAABB(positions)
 * Calcule un bounding box pour un ensemble de points.
 */
MeshSlicerCenter.prototype._computeAABB = function(positions) {
    var minx = +Infinity, miny = +Infinity, minz = +Infinity;
    var maxx = -Infinity, maxy = -Infinity, maxz = -Infinity;

    for (var i = 0; i < positions.length; i++) {
        var p = positions[i];
        if (p.x < minx) minx = p.x;
        if (p.y < miny) miny = p.y;
        if (p.z < minz) minz = p.z;

        if (p.x > maxx) maxx = p.x;
        if (p.y > maxy) maxy = p.y;
        if (p.z > maxz) maxz = p.z;
    }

    var center = new pc.Vec3(
        (minx + maxx) / 2,
        (miny + maxy) / 2,
        (minz + maxz) / 2
    );
    var halfExtents = new pc.Vec3(
        (maxx - minx) / 2,
        (maxy - miny) / 2,
        (maxz - minz) / 2
    );
    return new pc.BoundingBox(center, halfExtents);
};