var FlyCamera = pc.createScript('flyCamera');

FlyCamera.attributes.add('speed', {
    type: 'number',
    default: 10
});

FlyCamera.attributes.add('fastSpeed', {
    type: 'number',
    default: 20
});

FlyCamera.attributes.add('mode', {
    type: 'number',
    default: 0,
    enum: [
        { "Lock": 0 }, // pointeur souris verrouillé
        { "Drag": 1 }  // clic gauche+drag
    ]
});

// Initialise
FlyCamera.prototype.initialize = function () {
    // On récupère l’orientation initiale
    var eulers = this.entity.getLocalEulerAngles();
    this.ex = eulers.x;
    this.ey = eulers.y;

    // Pour éviter un « saut » initial lors du premier déplacement souris
    this.moved = false;

    // Pour gérer le bouton gauche en mode "Drag"
    this.lmbDown = false;

    // On désactive le menu contextuel (clic droit)
    this.app.mouse.disableContextMenu();

    // Écoute des événements souris
    this.app.mouse.on(pc.EVENT_MOUSEMOVE, this.onMouseMove, this);
    this.app.mouse.on(pc.EVENT_MOUSEDOWN, this.onMouseDown, this);
    this.app.mouse.on(pc.EVENT_MOUSEUP, this.onMouseUp, this);
};

// update (chaque frame)
FlyCamera.prototype.update = function (dt) {
    // Vérifie si l’utilisateur contrôle le mouvement (clavier/souris)
    var userIsControlling = this.isUserControlling();

    if (userIsControlling) {
        // Applique l’orientation
        this.entity.setLocalEulerAngles(this.ex, this.ey, 0);

        // Gère la translation (ZQSD, flèches, etc.)
        var app = this.app;
        var speed = (app.keyboard.isPressed(pc.KEY_SHIFT)) ? this.fastSpeed : this.speed;

        if (app.keyboard.isPressed(pc.KEY_UP) || app.keyboard.isPressed(pc.KEY_W)) {
            this.entity.translateLocal(0, 0, -speed * dt);
        } else if (app.keyboard.isPressed(pc.KEY_DOWN) || app.keyboard.isPressed(pc.KEY_S)) {
            this.entity.translateLocal(0, 0, speed * dt);
        }

        if (app.keyboard.isPressed(pc.KEY_LEFT) || app.keyboard.isPressed(pc.KEY_A)) {
            this.entity.translateLocal(-speed * dt, 0, 0);
        } else if (app.keyboard.isPressed(pc.KEY_RIGHT) || app.keyboard.isPressed(pc.KEY_D)) {
            this.entity.translateLocal(speed * dt, 0, 0);
        }
    } else {
        // Si l’utilisateur ne contrôle pas, on récupère l’orientation 
        // actuelle pour rester synchronisés
        var currentEulers = this.entity.getLocalEulerAngles();
        this.ex = currentEulers.x;
        this.ey = currentEulers.y;
    }
};

// Renvoie true si l'utilisateur est en train de contrôler la caméra (clavier/souris)
FlyCamera.prototype.isUserControlling = function() {
    var app = this.app;

    // 1) Clavier : ZQSD, flèches, SHIFT
    if (app.keyboard.isPressed(pc.KEY_W)   || app.keyboard.isPressed(pc.KEY_UP)    ||
        app.keyboard.isPressed(pc.KEY_S)   || app.keyboard.isPressed(pc.KEY_DOWN)  ||
        app.keyboard.isPressed(pc.KEY_A)   || app.keyboard.isPressed(pc.KEY_LEFT)  ||
        app.keyboard.isPressed(pc.KEY_D)   || app.keyboard.isPressed(pc.KEY_RIGHT) ||
        app.keyboard.isPressed(pc.KEY_SHIFT)) {
        return true;
    }

    // 2) Souris : 
    //    - mode "Lock" => on vérifie si le pointeur est verrouillé
    //    - mode "Drag" => on vérifie si le bouton gauche est enfoncé
    if (this.mode === 0) { // Lock
        if (pc.Mouse.isPointerLocked()) {
            return true;
        }
    } else { // Drag
        if (this.lmbDown) {
            return true;
        }
    }

    return false;
};

// Gère le mouvement de souris
FlyCamera.prototype.onMouseMove = function (event) {
    // Selon le mode :
    // - "Lock": il faut que le pointeur soit "pointerLocked"
    // - "Drag": il faut le bouton gauche enfoncé
    var needMouseControl = false;

    if (this.mode === 0) { // Lock
        if (pc.Mouse.isPointerLocked()) {
            needMouseControl = true;
        }
    } else { // Drag
        if (this.lmbDown) {
            needMouseControl = true;
        }
    }

    if (!needMouseControl)
        return;

    // Évite un saut la toute première fois
    if (!this.moved) {
        this.moved = true;
        return;
    }

    // Ajuste ex/ey en fonction du mouvement de souris
    this.ex -= event.dy / 5;
    this.ex = pc.math.clamp(this.ex, -90, 90);
    this.ey -= event.dx / 5;
};

// Bouton souris enfoncé
FlyCamera.prototype.onMouseDown = function (event) {
    if (event.button === pc.MOUSEBUTTON_LEFT) {
        this.lmbDown = true;

        // En mode "Lock", on tente de locker le pointeur
        if (this.mode === 0 && !pc.Mouse.isPointerLocked()) {
            this.app.mouse.enablePointerLock();
        }
    }
};

// Bouton souris relâché
FlyCamera.prototype.onMouseUp = function (event) {
    if (event.button === pc.MOUSEBUTTON_LEFT) {
        this.lmbDown = false;
    }
};
